/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.sharedmemory.inhouseservice.messenger;

import org.jssec.android.shared.SigPerm;
import org.jssec.android.shared.Utils;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.os.SharedMemory;
import android.system.ErrnoException;
import android.util.Log;
import android.widget.Toast;

import java.nio.ByteBuffer;

import static android.content.pm.PackageManager.PERMISSION_GRANTED;
import static android.system.OsConstants.PROT_EXEC;
import static android.system.OsConstants.PROT_READ;
import static android.system.OsConstants.PROT_WRITE;

public class SHMService extends Service {
    // In-house Signature Permission
    private static final String MY_PERMISSION =
        "org.jssec.android.sharedmemory.inhouseservice.messenger.MY_PERMISSION";

    // Hash value of the certificate of In-house applications
    private static String sMyCertHash = null;
    private static String myCertHash(Context context) {
        if (sMyCertHash == null) {
            if (Utils.isDebuggable(context)) {
                // Hash value of the certificate "androiddebugkey" stored in
                // debug.keystore
                sMyCertHash = "0EFB7236 328348A9 89718BAD DF57F544 D5CCB4AE B9DB34BC 1E29DD26 F77C8255";
            } else {
                // Hash value of the certificate "my company key" in keystore
                sMyCertHash = "D397D343 A5CBC10F 4EDDEB7C A10062DE 5690984F 1FB9E88B D7B3A7C2 42E142CA";
            }
        }
        return sMyCertHash;
    }

    private final String TAG = "SHM";

    // Strings which will be sent to client
    private final String greeting = "Hi! I send you my memory. Let's Share it!";
    // Page size is 4K bytes
    public static final int PAGE_SIZE = 1024 * 4;
    // In this example, we use two SharedMemory objects
    // Client side specify the one of these SharedMemory by using following
    // identify
    public static final int SHMEM1 = 0;
    public static final int SHMEM2 = 1;

    // Instances of Shared Memory
    // mSHMem1: used for sending data to client
    private SharedMemory mSHMem1 = null;
    // ByteBuffer for mapping mSHMem
    private ByteBuffer m1Buffer1;
    // mSHMem2: used for receiving data from client side
    private SharedMemory mSHMem2 = null;
    // ByteBuffer for mapping mSHMem2
    private ByteBuffer m2Buffer1;
    private ByteBuffer m2Buffer2;
    // true iff all ByteBuffers are mapped successfully
    private boolean mBufferMapped = false;


    // In this example, Messenger is used for communicating with client
    // The follwings are message identifier for the communication
    public static final int MSG_INVALID = Integer.MIN_VALUE;
    public static final int MSG_ATTACH  =
            MSG_INVALID + 1; // client requests SHMEM1
    public static final int MSG_ATTACH2 =
            MSG_ATTACH + 1;  // client requests SHMEM2
    public static final int MSG_DETACH  =
            MSG_ATTACH2 + 1; // client no more need SHMEM1
    public static final int MSG_DETACH2 =
            MSG_DETACH + 1;  // client no more need SHMEM2
    public static final int MSG_REPLY1  =
            MSG_DETACH2 + 1; // first reply from client
    public static final int MSG_REPLY2  =
            MSG_REPLY1 + 1;  // second reply from client
    public static final int MSG_END     =
            MSG_REPLY2 + 1;  // Service declared the end of the session

    // Handler manipulating Message received from client
    private class CommHandler extends Handler {
        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
            case MSG_ATTACH:
                Log.d(TAG, "got MSG_ATTACH");
                shareWith1(msg);
                break;
            case MSG_ATTACH2:
                Log.d(TAG, "got MSG_ATTACH2");
                shareWith2(msg);
                break;
            case MSG_DETACH:
                Log.d(TAG, "got MSG_DETACH");
                unShare(msg);
                break;
            case MSG_REPLY1:
                Log.d(TAG, "got MSG_REPLY1");
                gotReply(msg);
                break;
            case MSG_REPLY2:
                Log.d(TAG, "got MSG_REPLY2");
                gotReply2(msg);
                break;
            default:
                invalidMsg(msg);
            }
        }
    }

    private final Handler mHandler = new CommHandler();

    // Messenger used for receiving data from client
    private final Messenger mMessenger = new Messenger(mHandler);

    // When bound, extract Binfer from Message, pass it to client
    @Override
    public IBinder onBind(Intent intent) {
        // ** POINT 4 *** Verify that the in-house signature permission is defined
        // by an in-house application.
        if (!SigPerm.test(this, MY_PERMISSION, myCertHash(this))) {
            Toast.makeText(this, "In-house signature permission is not defined by an in-house application.", Toast.LENGTH_LONG).show();
            return null;
        }

        // *** POINT 5 *** Verify the safety of received Intent even if the Intent
        // was sent from an in-house application
        // Omitted because this is an sample code. Refer to
        // "3.2 Handling Input Data Carefully and Securely".
        String param = intent.getStringExtra("PARAM");
        Log.d(TAG, String.format("Received Parameter [%s]!", param));
        return mMessenger.getBinder();
    }

    // Mapping layout
    // Offset must be page boundary
    public static final int SHMEM1_BUF1_OFFSET = 0;
    public static final int SHMEM1_BUF1_LENGTH = 1024;
    public static final int SHMEM2_BUF1_OFFSET = 0;
    public static final int SHMEM2_BUF1_LENGTH = 1024;
    public static final int SHMEM2_BUF2_OFFSET = PAGE_SIZE;
    public static final int SHMEM2_BUF2_LENGTH = 128;

    // Allocate 2 SharedMemory objects
    private boolean allocateSharedMemory() {
        try {
            // For sending data to client
            mSHMem1 = SharedMemory.create("SHM", PAGE_SIZE);
            // For receiving data from client
            mSHMem2 = SharedMemory.create("SHM2", PAGE_SIZE * 2);
        } catch (ErrnoException e) {
            Log.e(TAG, "failed to allocate shared memory" + e.getMessage());
            return false;
        }
        return true;
    }

    // Map specified SharedMemory
    private ByteBuffer mapShared(SharedMemory mem,
                                 int prot, int offset, int size) {
        ByteBuffer tBuf ;
        try {
            tBuf = mem.map(prot, offset, size);
        } catch (ErrnoException e) {
            Log.e(TAG, "could not map, prot=" + prot + ", offset=" + offset + ", length=" + size + "\n " + e.getMessage() + "err no. = " + e.errno);
            return null;
        } catch (IllegalArgumentException e){
            Log.e(TAG, "map failed: " + e.getMessage());
            return null;
        }
        Log.d(TAG, "mmap success: prot=" + prot);
        return tBuf;
    }

    // Server side mappings of SharedMemory objects
    private void mapMemory() {
        // mSHMem1: read/write
        m1Buffer1 = mapShared(mSHMem1,
            PROT_READ | PROT_WRITE | PROT_EXEC, SHMEM1_BUF1_OFFSET,
            SHMEM1_BUF1_LENGTH);
        // mSHMem2: separate two regions, read/write for each
        m2Buffer1 = mapShared(mSHMem2,
            PROT_READ | PROT_WRITE, SHMEM2_BUF1_OFFSET,
            SHMEM2_BUF1_LENGTH);
        m2Buffer2 = mapShared(mSHMem2,
            PROT_READ | PROT_WRITE, SHMEM2_BUF2_OFFSET,
            SHMEM2_BUF2_LENGTH);

        if (m1Buffer1 != null && m2Buffer1 != null && m2Buffer2 != null) {
            mBufferMapped = true;
        }
    }

    // Free SharedMemory
    private void deAllocateSharedMemory () {
        if (mBufferMapped) {
            if (mSHMem1 != null) {
                if (m1Buffer1 != null) SharedMemory.unmap(m1Buffer1);
                m1Buffer1 = null;
                mSHMem1.close();
                mSHMem1 = null;
            }
            if (mSHMem2 != null) {
                if (m2Buffer1 != null) SharedMemory.unmap(m2Buffer1);
                if (m2Buffer2 != null) SharedMemory.unmap(m2Buffer2);
                m2Buffer1 = null;
                m2Buffer2 = null;
                mSHMem2.close();
                mSHMem2 = null;
            }
            mBufferMapped = false;
        }
    }

    @Override
    public void onCreate() {
        super.onCreate();

        // Allocate SharedMemory objects at the time of instantiation
        // If succeded, map SharedMemory objects
        if (allocateSharedMemory()) {
            mapMemory();
        }
    }

    // Provide SHMEM1 to client
    private void shareWith1(Message msg){
        // If failed in allocating or mapping, do nothing
        if (!mBufferMapped) return;

        // *** POINT 6 *** Before passing the shared memory on to a client, use
        // SharedMemory#setProtect() to limit the available operations by the
        // client.
        // Client can only read from mSHMem1
        mSHMem1.setProtect(PROT_READ);

        // Mapping hash been done before the above setProtect(PROT_READ),
        // so setver side can write to mShMem1 via m1Buffer1
        // Put the size of the messege, then add message string
        m1Buffer1.putInt(greeting.length());
        m1Buffer1.put(greeting.getBytes());

        try {
            // Pass the SharedMemory object to the client
            Message sMsg = Message.obtain(null, SHMEM1, mSHMem1);
            msg.replyTo.send(sMsg);
        } catch (RemoteException e) {
            Log.e(TAG, "Failed to share" + e.getMessage());
        }
    }

    // Provide SHMEM2
    private void shareWith2(Message msg) {
        if (!mBufferMapped) return;

        // *** POINT 6 *** Before passing the shared memory on to a client, use
        // SharedMemory#setProtect() to limit the available operations by the
        // client.
        // Client can write to mSHMem2
        mSHMem2.setProtect(PROT_WRITE);
        // Set messages to client in each buffer
        final String greeting2 = "You can write here!";
        m2Buffer1.putInt(greeting2.length());
        m2Buffer1.put(greeting2.getBytes());
        final String greeting3 = "From this point, I'll also write.";
        m2Buffer2.putInt(greeting3.length());
        m2Buffer2.put(greeting3.getBytes());
        try {
            // Pass the shared memory objects to the client
            Message sMsg = Message.obtain(null, SHMEM2, mSHMem2);
            msg.replyTo.send(sMsg);
        } catch (RemoteException e){
            Log.e(TAG, "failed to share mSHMem2" + e.getMessage());
        }
    }

    // Stop sharing memory
    private void unShare(Message msg){
        deAllocateSharedMemory();
    }

    // Accepted invalid message
    private void invalidMsg(Message msg){
        Log.e(TAG, "Got an Invalid message: " + msg.what);
    }

    // Retrive data which set by the client from buffer
    // The first element is a size of the data followed by byte sequence of a
    // string
    private String extractReply (ByteBuffer buf){
        int len = buf.getInt();
        byte [] bytes = new byte[len];
        buf.get(bytes);
        return new String(bytes);
    }

    // In this example, server side accepts two types of message from the client
    // goReply() assumes that  m1Buffer1 holds a data from the client
    private void gotReply(Message msg) {
        m1Buffer1.rewind();
        String message = extractReply(m1Buffer1);
        if (!message.equals(greeting)){
            Log.e(TAG, "my message was overwritten: " + message);
        }
    }
    // got Reply2() assumes m2Buffer1 holds a data from the client
    private void gotReply2(Message msg) {
        m2Buffer1.rewind();
        String message = extractReply(m2Buffer1);
        android.util.Log.d(TAG, "got a message of length " + message.length() +
                                 " from client: " + message);
        // Accepting a message in m2Buffer1 is a sign of the end of sharing memory
        Message eMsg = Message.obtain();
        eMsg.what = MSG_END;
        try {
            msg.replyTo.send(eMsg);
        } catch (RemoteException e){
            Log.e(TAG, "error in reply 2: " + e.getMessage());
        }
    }
}
